'use strict';
const mpd = require('mpd');
const cmd = mpd.cmd;

/**
 * Minimal plan→tracks resolver using MPD search.
 * Extensible: add BPM/mood/decade filtering if your tags carry them.
 */

function connect() {
  return new Promise((resolve, reject) => {
    const client = mpd.connect({ host: '127.0.0.1', port: 6600 });
    client.on('ready', () => resolve(client));
    client.on('error', reject);
  });
}

function parseKeyVals(str) {
  const lines = (str || '').split('\n');
  const files = [];
  let cur = null;
  for (const ln of lines) {
    if (!ln.trim()) continue;
    const idx = ln.indexOf(': ');
    if (idx < 0) continue;
    const k = ln.slice(0, idx);
    const v = ln.slice(idx + 2);
    if (k === 'file') {
      if (cur && cur.file) files.push(cur.file);
      cur = { file: v };
    } else if (cur) {
      cur[k] = v;
    }
  }
  if (cur && cur.file) files.push(cur.file);
  return files;
}

async function mpdSearch(tag, value) {
  const c = await connect();
  const msg = await new Promise((resolve, reject) =>
    c.sendCommand(cmd('search', [tag, value]), (e, m) => e ? reject(e) : resolve(m))
  );
  c.close();
  return parseKeyVals(msg);
}

async function findTracks(plan, limit, logger) {
  const chosen = new Set();

  // Genres
  if (plan.filters?.genres?.length) {
    for (const g of plan.filters.genres) {
      const hits = await mpdSearch('genre', g);
      hits.forEach(u => chosen.add(u));
    }
  }

  // Seed artists / titles
  if (plan.seed_tracks?.length) {
    for (const st of plan.seed_tracks) {
      if (st.artist) {
        const hits = await mpdSearch('artist', st.artist);
        hits.forEach(u => chosen.add(u));
      }
      if (st.title) {
        const hits = await mpdSearch('title', st.title);
        hits.forEach(u => chosen.add(u));
      }
    }
  }

  // Moods / decades / BPM — optional: implement tag-based refinement here
  // For now we rely on genre+artist seeds.

  const out = Array.from(chosen).slice(0, limit || 60);
  logger.info(`[LocalAI] Selector picked ${out.length} tracks`);
  // MPD file paths are appropriate URIs for 'mpd' service via addQueueItems
  return out.map(u => 'music-library/' + u);
}

module.exports = { findTracks };
