'use strict';
const axios = require('axios');

const SYSTEM_PROMPT =
  'You are a playlist planning assistant. Return ONLY compact JSON matching this schema:\n' +
  '{ "filters": { "genres":[string], "moods":[string], "decades":[string], "min_bpm": number|null, "max_bpm": number|null }, ' +
  '"seed_tracks":[{"artist":string,"title":string|null}], "max_tracks": number }\n' +
  'No commentary, no markdown, no code fences.';

function tryJson(s) {
  try { return JSON.parse(s); } catch { return null; }
}

async function planFromPrompt(llmCfg, userPrompt, maxTracks, logger) {
  const prompt = `User request: ${userPrompt}\nReturn JSON with max_tracks=${maxTracks}. Use filters and seed_tracks wisely.`;

  if (llmCfg.mode === 'ollama') {
    const url = llmCfg.baseUrl.replace(/\/$/, '') + '/api/generate';
    const res = await axios.post(url, {
      model: llmCfg.model,
      prompt: `${SYSTEM_PROMPT}\n\n${prompt}`,
      stream: false
    }, { timeout: llmCfg.timeoutMs });

    const text = (res && res.data && res.data.response) ? String(res.data.response).trim() : '';
    const json = tryJson(text);
    if (!json) throw new Error('LLM did not return valid JSON');
    logger.info('[LocalAI] LLM plan received');
    return json;
  } else {
    // OpenAI-compatible /v1/chat/completions
    const url = llmCfg.baseUrl.replace(/\/$/, '') + '/v1/chat/completions';
    const headers = llmCfg.apiKey ? { Authorization: 'Bearer ' + llmCfg.apiKey } : {};
    const res = await axios.post(url, {
      model: llmCfg.model,
      messages: [
        { role: 'system', content: SYSTEM_PROMPT },
        { role: 'user', content: prompt }
      ],
      temperature: 0.2
    }, { timeout: llmCfg.timeoutMs, headers });

    const text = res?.data?.choices?.[0]?.message?.content?.trim() || '';
    const json = tryJson(text);
    if (!json) throw new Error('LLM did not return valid JSON');
    logger.info('[LocalAI] LLM plan received (OpenAI-compatible)');
    return json;
  }
}

module.exports = { planFromPrompt };
