'use strict';

const libQ = require('kew');
const VConf = require('v-conf');

const llm = require('./lib/llmClient');
const selector = require('./lib/selector');
const queue = require('./lib/queue');
const art = require('./lib/artClient');

module.exports = ControllerLocalAI;

function ControllerLocalAI(context) {
  this.context = context;
  this.commandRouter = context.coreCommand;
  this.logger = context.logger;
  this.configManager = context.configManager;

  this.config = new VConf();
}

ControllerLocalAI.prototype.getConfigurationFiles = function () {
  return ['config.json'];
};

ControllerLocalAI.prototype.onVolumioStart = function () {
  const defer = libQ.defer();
  try {
    const configFile = this.commandRouter.pluginManager.getConfigurationFile(this.context, 'config.json');
    this.config.loadFile(configFile);
    defer.resolve();
  } catch (e) {
    this.logger.error('[LocalAI] onVolumioStart config load failed: ' + e);
    defer.reject(e);
  }
  return defer.promise;
};

ControllerLocalAI.prototype.onStart = function () {
  const defer = libQ.defer();
  try {
    // Add browse source (optional convenience entry)
    this.commandRouter.volumioAddToBrowseSources({
      name: 'AI Playlist',
      uri: 'ai_playlist',
      plugin_type: 'music_service',
      plugin_name: 'local-ai-playlists',
      albumart: '/albumart?sourceicon=assets/icon.png'
    });
    this.logger.info('[LocalAI] Started');
    defer.resolve();
  } catch (e) {
    this.logger.error('[LocalAI] onStart error: ' + e);
    defer.reject(e);
  }
  return defer.promise;
};

ControllerLocalAI.prototype.onStop = function () {
  const defer = libQ.defer();
  try {
    this.commandRouter.volumioRemoveToBrowseSources('AI Playlist');
    this.logger.info('[LocalAI] Stopped');
    defer.resolve();
  } catch (e) {
    this.logger.error('[LocalAI] onStop error: ' + e);
    defer.reject(e);
  }
  return defer.promise;
};

ControllerLocalAI.prototype.getUIConfig = function () {
  const defer = libQ.defer();
  const langCode = this.commandRouter.sharedVars.get('language_code') || 'en';

  this.commandRouter.i18nJson(
    __dirname + '/i18n/strings_' + langCode + '.json',
    __dirname + '/i18n/strings_en.json',
    __dirname + '/UIConfig.json'
  ).then(uiconf => {
    // Populate with current config values
    // LLM
    uiconf.sections[0].content.find(x => x.id === 'llm_mode').value.value = this.config.get('llm.mode', 'ollama');
    uiconf.sections[0].content.find(x => x.id === 'llm_mode').value.label = this.config.get('llm.mode', 'ollama') === 'openai' ? 'OpenAI-compatible' : 'Ollama';
    uiconf.sections[0].content.find(x => x.id === 'llm_base').value = this.config.get('llm.baseUrl', 'http://127.0.0.1:11434');
    uiconf.sections[0].content.find(x => x.id === 'llm_model').value = this.config.get('llm.model', 'llama3:8b-instruct');
    uiconf.sections[0].content.find(x => x.id === 'llm_timeout').value = String(this.config.get('llm.timeoutMs', 30000));
    uiconf.sections[0].content.find(x => x.id === 'llm_api_key').value = this.config.get('llm.apiKey', '');

    // Artwork
    uiconf.sections[1].content.find(x => x.id === 'art_enabled').value = this.config.get('art.enabled', false);
    uiconf.sections[1].content.find(x => x.id === 'art_base').value = this.config.get('art.baseUrl', 'http://127.0.0.1:7860');
    uiconf.sections[1].content.find(x => x.id === 'art_size').value = this.config.get('art.size', '1024x1024');

    // Defaults for prompt UI
    uiconf.sections[2].content.find(x => x.id === 'user_prompt').value = '';
    uiconf.sections[2].content.find(x => x.id === 'max_tracks').value = '60';

    return uiconf;
  }).then(uiconf => defer.resolve(uiconf))
    .fail(err => {
      this.logger.error('[LocalAI] getUIConfig error: ' + err);
      defer.reject(err);
    });

  return defer.promise;
};

// Handle settings save (section 0 & 1)
ControllerLocalAI.prototype.saveSettings = function (data) {
  const defer = libQ.defer();
  try {
    // LLM
    this.config.set('llm.mode', data.llm_mode && data.llm_mode.value ? data.llm_mode.value : 'ollama');
    this.config.set('llm.baseUrl', data.llm_base || 'http://127.0.0.1:11434');
    this.config.set('llm.model', data.llm_model || 'llama3:8b-instruct');
    this.config.set('llm.timeoutMs', parseInt(data.llm_timeout || '30000', 10));
    this.config.set('llm.apiKey', data.llm_api_key || '');

    // Artwork
    this.config.set('art.enabled', !!data.art_enabled);
    this.config.set('art.baseUrl', data.art_base || 'http://127.0.0.1:7860');
    this.config.set('art.size', data.art_size || '1024x1024');

    this.commandRouter.pushToastMessage('success', 'Local AI', 'Settings saved');
    defer.resolve();
  } catch (e) {
    this.logger.error('[LocalAI] saveSettings error: ' + e);
    this.commandRouter.pushToastMessage('error', 'Local AI', 'Failed to save settings');
    defer.reject(e);
  }
  return defer.promise;
};

// Generate playlist (section 2 button)
ControllerLocalAI.prototype.generatePlaylist = function (data) {
  const defer = libQ.defer();

  const prompt = (data && data.user_prompt) ? String(data.user_prompt).trim() : '';
  const maxTracks = Math.max(1, parseInt((data && data.max_tracks) ? data.max_tracks : '60', 10));
  if (!prompt) {
    this.commandRouter.pushToastMessage('error', 'Local AI', 'Please enter a prompt');
    return libQ.resolve();
  }

  const cfg = {
    llm: {
      mode: this.config.get('llm.mode', 'ollama'),
      baseUrl: this.config.get('llm.baseUrl', 'http://127.0.0.1:11434'),
      model: this.config.get('llm.model', 'llama3:8b-instruct'),
      timeoutMs: this.config.get('llm.timeoutMs', 30000),
      apiKey: this.config.get('llm.apiKey', '')
    },
    art: {
      enabled: this.config.get('art.enabled', false),
      baseUrl: this.config.get('art.baseUrl', 'http://127.0.0.1:7860'),
      size: this.config.get('art.size', '1024x1024')
    }
  };

  this.logger.info(`[LocalAI] Generating playlist for prompt: "${prompt}"`);

  llm.planFromPrompt(cfg.llm, prompt, maxTracks, this.logger)
    .then(plan => selector.findTracks(plan, maxTracks, this.logger))
    .then(trackUris => {
      if (!trackUris.length) throw new Error('No matching tracks found in your library');

      // Build queue items for Volumio (MPD service for local files)
      const items = trackUris.map(u => ({ service: 'mpd', uri: u }));

      return queue.replaceQueueAndPlay(this.commandRouter, items)
        .then(() => ({ count: items.length }));
    })
    .then(async ({ count }) => {
      // Optional artwork generation (saved to /data)
      if (cfg.art.enabled) {
        try {
          const imgPath = await art.generate(cfg.art, prompt, this.logger);
          this.logger.info('[LocalAI] Artwork saved: ' + imgPath);
        } catch (e) {
          this.logger.warn('[LocalAI] Artwork generation failed: ' + e);
        }
      }
      this.commandRouter.pushToastMessage('success', 'Local AI', `Created & playing: ${count} tracks`);
      defer.resolve();
    })
    .fail(err => {
      this.logger.error('[LocalAI] generatePlaylist error: ' + err);
      this.commandRouter.pushToastMessage('error', 'Local AI', (err && err.message) ? err.message : 'Failed to generate playlist');
      defer.resolve(); // resolve to close modal gracefully
    });

  return defer.promise;
};

// Volumio calls this for global search; keep it safe and fast.
ControllerLocalAI.prototype.search = function (query) {
  return libQ.resolve([]);
};

// Optional: simple Browse source
ControllerLocalAI.prototype.handleBrowseUri = function (curUri) {
  const defer = libQ.defer();

  if (curUri === 'ai_playlist') {
    defer.resolve({
      navigation: {
        prev: { uri: 'music-library' },
        lists: [{
          title: 'AI Playlist Generator',
          icon: 'fa-magic',
          availableListViews: ['list'],
          items: [
            {
              service: 'local-ai-playlists',
              type: 'item-no-menu',
              title: 'Open plugin settings to generate a playlist',
              icon: 'fa-external-link',
              uri: 'ai_playlist/open'
            }
          ]
        }]
      }
    });
  } else {
    defer.reject(new Error('Unknown URI: ' + curUri));
  }

  return defer.promise;
};
